"""
Database SQLite per Pianificazione Acquedotto
"""
import sqlite3
import os
from contextlib import contextmanager

DATABASE_PATH = os.getenv("DATABASE_PATH", "/data/planning.db")

def get_db_path():
    """Ritorna il path del database, creando la directory se necessario"""
    db_dir = os.path.dirname(DATABASE_PATH)
    if db_dir and not os.path.exists(db_dir):
        os.makedirs(db_dir)
    return DATABASE_PATH

@contextmanager
def get_connection():
    """Context manager per connessione database"""
    conn = sqlite3.connect(get_db_path())
    conn.row_factory = sqlite3.Row
    try:
        yield conn
        conn.commit()
    finally:
        conn.close()

def init_database():
    """Inizializza il database con le tabelle necessarie"""
    with get_connection() as conn:
        cursor = conn.cursor()
        
        # Tabella managers
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS managers (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                password TEXT NOT NULL,
                color TEXT DEFAULT '#0891b2',
                is_superuser INTEGER DEFAULT 0
            )
        """)
        
        # Tabella team_assignments (persone assegnate ai manager)
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS team_assignments (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                person_name TEXT NOT NULL,
                manager_id TEXT NOT NULL,
                FOREIGN KEY (manager_id) REFERENCES managers(id),
                UNIQUE(person_name, manager_id)
            )
        """)
        
        # Tabella planning
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS planning (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                person TEXT NOT NULL,
                date TEXT NOT NULL,
                period TEXT NOT NULL,
                activity TEXT,
                manager_id TEXT,
                UNIQUE(person, date, period)
            )
        """)
        
        # Tabella daily_notes
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS daily_notes (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                note TEXT NOT NULL,
                manager_id TEXT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (manager_id) REFERENCES managers(id)
            )
        """)
        
        # Inserisci dati di esempio se il database è vuoto
        cursor.execute("SELECT COUNT(*) FROM managers")
        if cursor.fetchone()[0] == 0:
            insert_sample_data(cursor)
        
        conn.commit()
        print(f"✅ Database inizializzato: {get_db_path()}")

def insert_sample_data(cursor):
    """Inserisce dati di esempio per iniziare"""
    # Manager di esempio
    managers = [
        ("marco", "Marco Rossi", "admin123", "#0891b2", 1),  # Superuser
        ("luigi", "Luigi Bianchi", "pass123", "#059669", 0),
        ("anna", "Anna Verdi", "pass123", "#7c3aed", 0),
    ]
    
    cursor.executemany(
        "INSERT INTO managers (id, name, password, color, is_superuser) VALUES (?, ?, ?, ?, ?)",
        managers
    )
    
    # Persone nei team
    team_assignments = [
        # Team Marco
        ("Mario Russo", "marco"),
        ("Giuseppe Esposito", "marco"),
        ("Antonio Romano", "marco"),
        # Team Luigi
        ("Francesco Colombo", "luigi"),
        ("Alessandro Ricci", "luigi"),
        ("Andrea Marino", "luigi"),
        # Team Anna
        ("Luca Greco", "anna"),
        ("Paolo Ferrari", "anna"),
        ("Davide Costa", "anna"),
    ]
    
    cursor.executemany(
        "INSERT INTO team_assignments (person_name, manager_id) VALUES (?, ?)",
        team_assignments
    )
    
    print("✅ Dati di esempio inseriti")

# ============ CRUD MANAGERS ============

def get_all_managers():
    """Ritorna tutti i manager"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM managers ORDER BY name")
        return [dict(row) for row in cursor.fetchall()]

def get_manager_by_id(manager_id: str):
    """Ritorna un manager specifico"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM managers WHERE id = ?", (manager_id,))
        row = cursor.fetchone()
        return dict(row) if row else None

def create_manager(id: str, name: str, password: str, color: str = "#0891b2", is_superuser: int = 0):
    """Crea un nuovo manager"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "INSERT INTO managers (id, name, password, color, is_superuser) VALUES (?, ?, ?, ?, ?)",
            (id, name, password, color, is_superuser)
        )
        return True

def update_manager(id: str, name: str = None, password: str = None, color: str = None, is_superuser: int = None):
    """Aggiorna un manager"""
    with get_connection() as conn:
        cursor = conn.cursor()
        updates = []
        values = []
        
        if name is not None:
            updates.append("name = ?")
            values.append(name)
        if password is not None:
            updates.append("password = ?")
            values.append(password)
        if color is not None:
            updates.append("color = ?")
            values.append(color)
        if is_superuser is not None:
            updates.append("is_superuser = ?")
            values.append(is_superuser)
        
        if updates:
            values.append(id)
            cursor.execute(f"UPDATE managers SET {', '.join(updates)} WHERE id = ?", values)
        return True

def delete_manager(id: str):
    """Elimina un manager"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("DELETE FROM team_assignments WHERE manager_id = ?", (id,))
        cursor.execute("DELETE FROM managers WHERE id = ?", (id,))
        return True

# ============ CRUD TEAM ASSIGNMENTS ============

def get_teams():
    """Ritorna tutti i team raggruppati per manager"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM team_assignments ORDER BY manager_id, person_name")
        
        teams = {}
        for row in cursor.fetchall():
            manager_id = row['manager_id']
            if manager_id not in teams:
                teams[manager_id] = []
            teams[manager_id].append(row['person_name'])
        
        return teams

def get_all_team_assignments():
    """Ritorna tutte le assegnazioni"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("""
            SELECT ta.*, m.name as manager_name 
            FROM team_assignments ta 
            LEFT JOIN managers m ON ta.manager_id = m.id
            ORDER BY ta.manager_id, ta.person_name
        """)
        return [dict(row) for row in cursor.fetchall()]

def add_person_to_team(person_name: str, manager_id: str):
    """Aggiunge una persona a un team"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "INSERT OR REPLACE INTO team_assignments (person_name, manager_id) VALUES (?, ?)",
            (person_name, manager_id)
        )
        return True

def remove_person_from_team(person_name: str, manager_id: str):
    """Rimuove una persona da un team"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "DELETE FROM team_assignments WHERE person_name = ? AND manager_id = ?",
            (person_name, manager_id)
        )
        return True

def update_person_in_team(old_name: str, new_name: str, manager_id: str):
    """Aggiorna il nome di una persona"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE team_assignments SET person_name = ? WHERE person_name = ? AND manager_id = ?",
            (new_name, old_name, manager_id)
        )
        # Aggiorna anche il planning
        cursor.execute(
            "UPDATE planning SET person = ? WHERE person = ?",
            (new_name, old_name)
        )
        return True

# ============ CRUD PLANNING ============

def get_all_planning():
    """Ritorna tutto il planning"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM planning ORDER BY date, person, period")
        return [dict(row) for row in cursor.fetchall()]

def save_planning(person: str, date: str, period: str, activity: str, manager_id: str):
    """Salva o aggiorna una entry del planning"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO planning (person, date, period, activity, manager_id)
            VALUES (?, ?, ?, ?, ?)
            ON CONFLICT(person, date, period) 
            DO UPDATE SET activity = ?, manager_id = ?
        """, (person, date, period, activity, manager_id, activity, manager_id))
        return True

def delete_planning(person: str, date: str, period: str):
    """Cancella una entry del planning"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "DELETE FROM planning WHERE person = ? AND date = ? AND period = ?",
            (person, date, period)
        )
        return True

# ============ CRUD DAILY NOTES ============

def get_all_daily_notes():
    """Ritorna tutte le note"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("""
            SELECT dn.*, m.name as manager_name 
            FROM daily_notes dn 
            LEFT JOIN managers m ON dn.manager_id = m.id
            ORDER BY dn.date DESC, dn.created_at DESC
        """)
        return [dict(row) for row in cursor.fetchall()]

def create_daily_note(date: str, note: str, manager_id: str):
    """Crea una nuova nota"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "INSERT INTO daily_notes (date, note, manager_id) VALUES (?, ?, ?)",
            (date, note, manager_id)
        )
        return cursor.lastrowid

def update_daily_note(note_id: int, note: str):
    """Aggiorna una nota"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE daily_notes SET note = ? WHERE id = ?",
            (note, note_id)
        )
        return True

def delete_daily_note(note_id: int):
    """Cancella una nota"""
    with get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute("DELETE FROM daily_notes WHERE id = ?", (note_id,))
        return True

# Inizializza il database all'import
if __name__ == "__main__":
    init_database()
