"""
Pianificazione Acquedotto API - Versione SQLite
Backend FastAPI con database SQLite locale
"""
from fastapi import FastAPI, HTTPException, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import HTMLResponse
from fastapi.staticfiles import StaticFiles
from pydantic import BaseModel
from typing import Optional, List
import os

import database as db

app = FastAPI(title="Pianificazione Acquedotto API")

# CORS - permetti tutte le origini per sviluppo
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Inizializza database all'avvio
@app.on_event("startup")
async def startup():
    db.init_database()

# ============ MODELLI PYDANTIC ============

class PlanningEntry(BaseModel):
    person: str
    date: str
    period: str
    activity: str
    manager_id: str

class LoginRequest(BaseModel):
    manager_id: str
    password: str

class DailyNoteRequest(BaseModel):
    date: str
    note: str
    manager_id: str

# ============ API ENDPOINTS ============

@app.get("/")
async def root():
    return {
        "status": "online", 
        "service": "Pianificazione Acquedotto API", 
        "version": "2.0-SQLite",
        "database": "SQLite"
    }

@app.get("/api/managers")
async def get_managers():
    """Ritorna TUTTI i manager con i loro team"""
    all_managers = db.get_all_managers()
    teams = db.get_teams()
    
    return {
        "managers": [
            {
                "id": m['id'], 
                "name": m['name'], 
                "color": m['color'],
                "is_superuser": bool(m['is_superuser']),
                "team": teams.get(m['id'], [])
            } for m in all_managers
        ]
    }

@app.post("/api/login")
async def login(request: LoginRequest):
    """Login manager"""
    manager = db.get_manager_by_id(request.manager_id)
    
    if not manager or str(manager['password']) != request.password:
        raise HTTPException(status_code=401, detail="Credenziali non valide")
    
    teams = db.get_teams()
    
    return {
        "success": True, 
        "manager": {
            "id": manager['id'], 
            "name": manager['name'], 
            "color": manager['color'],
            "is_superuser": bool(manager['is_superuser']),
            "team": teams.get(manager['id'], [])
        }
    }

@app.get("/api/planning")
async def get_planning():
    """Ritorna tutto il planning"""
    results = db.get_all_planning()
    return {"results": results}

@app.post("/api/planning")
async def save_planning(entry: PlanningEntry):
    """Salva o aggiorna una cella del planning"""
    print(f"SAVE planning - person={entry.person}, date={entry.date}, period={entry.period}")
    db.save_planning(
        person=entry.person,
        date=entry.date,
        period=entry.period,
        activity=entry.activity,
        manager_id=entry.manager_id
    )
    return {"success": True}

@app.delete("/api/planning")
async def delete_planning(person: str, date: str, period: str):
    """Cancella una cella specifica"""
    print(f"DELETE planning - person={person}, date={date}, period={period}")
    db.delete_planning(person, date, period)
    return {"success": True}

@app.get("/api/daily-notes")
async def get_daily_notes():
    """Ritorna TUTTE le note giornaliere"""
    results = db.get_all_daily_notes()
    return {"results": results}

@app.post("/api/daily-notes")
async def create_daily_note(request: DailyNoteRequest):
    """Crea NUOVA nota"""
    print(f"CREATE daily note - date={request.date}, manager={request.manager_id}")
    note_id = db.create_daily_note(
        date=request.date,
        note=request.note,
        manager_id=request.manager_id
    )
    return {"id": note_id, "success": True}

@app.put("/api/daily-notes/{note_id}")
async def update_daily_note(note_id: int, request: DailyNoteRequest):
    """Aggiorna una nota esistente"""
    print(f"UPDATE daily note - id={note_id}")
    db.update_daily_note(note_id, request.note)
    return {"success": True}

@app.delete("/api/daily-notes/{note_id}")
async def delete_daily_note(note_id: int):
    """Cancella una nota specifica"""
    print(f"DELETE daily note - id={note_id}")
    db.delete_daily_note(note_id)
    return {"success": True}


# ============ ADMIN PANEL (Solo Superuser) ============

def verify_superuser(manager_id: str, password: str) -> bool:
    """Verifica che l'utente sia superuser"""
    manager = db.get_manager_by_id(manager_id)
    if not manager:
        return False
    if str(manager['password']) != password:
        return False
    return bool(manager['is_superuser'])

ADMIN_HTML = """
<!DOCTYPE html>
<html lang="it">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin - Pianificazione Acquedotto</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: linear-gradient(135deg, #1e3a5f 0%, #0f172a 100%);
            min-height: 100vh;
        }
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 10px 12px; text-align: left; border-bottom: 1px solid #e2e8f0; }
        th { background: #f1f5f9; font-weight: 600; color: #475569; }
        tr:hover { background: #f8fafc; }
        .btn {
            padding: 6px 12px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
        }
        .btn-primary { background: #0891b2; color: white; }
        .btn-primary:hover { background: #0e7490; }
        .btn-danger { background: #ef4444; color: white; }
        .btn-danger:hover { background: #dc2626; }
        .btn-success { background: #22c55e; color: white; }
        .btn-success:hover { background: #16a34a; }
        input, select {
            padding: 8px 12px;
            border: 1px solid #e2e8f0;
            border-radius: 6px;
            outline: none;
            transition: all 0.2s;
        }
        input:focus, select:focus {
            border-color: #0891b2;
            box-shadow: 0 0 0 3px rgba(8, 145, 178, 0.1);
        }
        .tab { 
            padding: 12px 24px; 
            cursor: pointer; 
            border-bottom: 3px solid transparent;
            color: #64748b;
            font-weight: 500;
        }
        .tab:hover { color: #0891b2; }
        .tab.active { 
            border-bottom-color: #0891b2; 
            color: #0891b2;
        }
        .badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
        }
        .badge-admin { background: #fef3c7; color: #92400e; }
        .badge-user { background: #e0f2fe; color: #0369a1; }
    </style>
</head>
<body>
    <div id="app" class="p-4 md:p-8">
        <!-- Login -->
        <div id="login-section" class="min-h-screen flex items-center justify-center">
            <div class="card p-8 w-full max-w-md">
                <div class="text-center mb-6">
                    <div class="text-4xl mb-2">🔐</div>
                    <h1 class="text-2xl font-bold text-slate-800">Admin Panel</h1>
                    <p class="text-slate-500 text-sm mt-1">Solo superuser autorizzati</p>
                </div>
                <form id="login-form" class="space-y-4">
                    <input type="text" id="admin-id" placeholder="ID Manager" class="w-full" required>
                    <input type="password" id="admin-password" placeholder="Password" class="w-full" required>
                    <button type="submit" class="btn btn-primary w-full py-3">Accedi come Admin</button>
                </form>
                <div id="login-error" class="text-red-500 text-sm text-center mt-4 hidden"></div>
                <div class="text-center mt-6">
                    <a href="/" class="text-cyan-600 text-sm hover:underline">← Torna all'app</a>
                </div>
            </div>
        </div>

        <!-- Admin Panel -->
        <div id="admin-section" class="hidden">
            <div class="card p-6 mb-6">
                <div class="flex flex-wrap items-center justify-between gap-4">
                    <div class="flex items-center gap-3">
                        <span class="text-3xl">⚙️</span>
                        <div>
                            <h1 class="text-xl font-bold text-slate-800">Pannello Amministrazione</h1>
                            <p class="text-slate-500 text-sm">Gestione database SQLite</p>
                        </div>
                    </div>
                    <div class="flex items-center gap-4">
                        <span id="admin-name" class="text-sm text-slate-600"></span>
                        <button onclick="logout()" class="btn btn-danger">Esci</button>
                    </div>
                </div>
            </div>

            <!-- Tabs -->
            <div class="card mb-6">
                <div class="flex border-b">
                    <div class="tab active" data-tab="managers" onclick="switchTab('managers')">👥 Manager</div>
                    <div class="tab" data-tab="teams" onclick="switchTab('teams')">👤 Persone/Team</div>
                    <div class="tab" data-tab="planning" onclick="switchTab('planning')">📅 Planning</div>
                    <div class="tab" data-tab="notes" onclick="switchTab('notes')">📌 Note</div>
                </div>
            </div>

            <!-- Tab Content: Managers -->
            <div id="tab-managers" class="tab-content card p-6">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-slate-800">Gestione Manager</h2>
                    <button onclick="showAddManager()" class="btn btn-success">+ Nuovo Manager</button>
                </div>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nome</th>
                            <th>Password</th>
                            <th>Colore</th>
                            <th>Ruolo</th>
                            <th>Azioni</th>
                        </tr>
                    </thead>
                    <tbody id="managers-table"></tbody>
                </table>
            </div>

            <!-- Tab Content: Teams -->
            <div id="tab-teams" class="tab-content card p-6 hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-slate-800">Gestione Persone nei Team</h2>
                    <button onclick="showAddPerson()" class="btn btn-success">+ Nuova Persona</button>
                </div>
                <table>
                    <thead>
                        <tr>
                            <th>Nome Persona</th>
                            <th>Manager</th>
                            <th>Azioni</th>
                        </tr>
                    </thead>
                    <tbody id="teams-table"></tbody>
                </table>
            </div>

            <!-- Tab Content: Planning -->
            <div id="tab-planning" class="tab-content card p-6 hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-slate-800">Visualizza Planning</h2>
                    <div class="flex gap-2">
                        <input type="date" id="filter-date" onchange="loadPlanning()" class="text-sm">
                        <button onclick="clearPlanningFilter()" class="btn btn-primary">Mostra tutto</button>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Persona</th>
                                <th>Data</th>
                                <th>Periodo</th>
                                <th>Attività</th>
                                <th>Manager</th>
                                <th>Azioni</th>
                            </tr>
                        </thead>
                        <tbody id="planning-table"></tbody>
                    </table>
                </div>
                <div id="planning-count" class="text-sm text-slate-500 mt-4"></div>
            </div>

            <!-- Tab Content: Notes -->
            <div id="tab-notes" class="tab-content card p-6 hidden">
                <h2 class="text-lg font-bold text-slate-800 mb-4">Note Giornaliere</h2>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Data</th>
                            <th>Nota</th>
                            <th>Autore</th>
                            <th>Azioni</th>
                        </tr>
                    </thead>
                    <tbody id="notes-table"></tbody>
                </table>
            </div>
        </div>

        <!-- Modal per aggiunta/modifica -->
        <div id="modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50">
            <div class="card p-6 w-full max-w-md mx-4">
                <div class="flex justify-between items-center mb-4">
                    <h3 id="modal-title" class="text-lg font-bold text-slate-800"></h3>
                    <button onclick="closeModal()" class="text-slate-400 hover:text-slate-600 text-2xl">&times;</button>
                </div>
                <form id="modal-form" class="space-y-4"></form>
            </div>
        </div>
    </div>

    <script>
        const API_URL = window.location.origin + '/api';
        let currentAdmin = null;
        let managersData = [];

        // ============ AUTH ============
        document.getElementById('login-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            const id = document.getElementById('admin-id').value;
            const password = document.getElementById('admin-password').value;
            
            try {
                const res = await fetch(`${API_URL}/admin/verify`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ manager_id: id, password })
                });
                
                if (res.ok) {
                    const data = await res.json();
                    currentAdmin = { id, password, name: data.name };
                    document.getElementById('admin-name').textContent = `👤 ${data.name}`;
                    document.getElementById('login-section').classList.add('hidden');
                    document.getElementById('admin-section').classList.remove('hidden');
                    loadAllData();
                } else {
                    showLoginError('Accesso negato. Solo superuser.');
                }
            } catch (err) {
                showLoginError('Errore di connessione');
            }
        });

        function showLoginError(msg) {
            const el = document.getElementById('login-error');
            el.textContent = msg;
            el.classList.remove('hidden');
        }

        function logout() {
            currentAdmin = null;
            document.getElementById('admin-section').classList.add('hidden');
            document.getElementById('login-section').classList.remove('hidden');
            document.getElementById('admin-id').value = '';
            document.getElementById('admin-password').value = '';
        }

        // ============ TABS ============
        function switchTab(tabName) {
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
            document.querySelectorAll('.tab-content').forEach(c => c.classList.add('hidden'));
            document.getElementById(`tab-${tabName}`).classList.remove('hidden');
        }

        // ============ LOAD DATA ============
        async function loadAllData() {
            await loadManagers();
            await loadTeams();
            await loadPlanning();
            await loadNotes();
        }

        async function loadManagers() {
            const res = await fetch(`${API_URL}/admin/managers`, {
                headers: adminHeaders()
            });
            const data = await res.json();
            managersData = data.managers || [];
            
            const tbody = document.getElementById('managers-table');
            tbody.innerHTML = managersData.map(m => `
                <tr>
                    <td><code class="bg-slate-100 px-2 py-1 rounded">${m.id}</code></td>
                    <td>${m.name}</td>
                    <td><code class="bg-slate-100 px-2 py-1 rounded text-xs">${m.password}</code></td>
                    <td><span style="background:${m.color}; width:20px; height:20px; display:inline-block; border-radius:4px;"></span> ${m.color}</td>
                    <td><span class="badge ${m.is_superuser ? 'badge-admin' : 'badge-user'}">${m.is_superuser ? 'SUPERUSER' : 'Manager'}</span></td>
                    <td>
                        <button onclick="editManager('${m.id}')" class="btn btn-primary mr-2">Modifica</button>
                        <button onclick="deleteManager('${m.id}')" class="btn btn-danger">Elimina</button>
                    </td>
                </tr>
            `).join('');
        }

        async function loadTeams() {
            const res = await fetch(`${API_URL}/admin/teams`, {
                headers: adminHeaders()
            });
            const data = await res.json();
            
            const tbody = document.getElementById('teams-table');
            tbody.innerHTML = (data.assignments || []).map(a => `
                <tr>
                    <td>${a.person_name}</td>
                    <td><span class="text-cyan-600 font-medium">${a.manager_name || a.manager_id}</span></td>
                    <td>
                        <button onclick="editPerson('${a.person_name}', '${a.manager_id}')" class="btn btn-primary mr-2">Modifica</button>
                        <button onclick="deletePerson('${a.person_name}', '${a.manager_id}')" class="btn btn-danger">Elimina</button>
                    </td>
                </tr>
            `).join('');
        }

        async function loadPlanning() {
            const filterDate = document.getElementById('filter-date').value;
            let url = `${API_URL}/admin/planning`;
            if (filterDate) url += `?date=${filterDate}`;
            
            const res = await fetch(url, { headers: adminHeaders() });
            const data = await res.json();
            
            const tbody = document.getElementById('planning-table');
            const items = data.planning || [];
            tbody.innerHTML = items.slice(0, 100).map(p => `
                <tr>
                    <td>${p.id}</td>
                    <td>${p.person}</td>
                    <td>${p.date}</td>
                    <td>${p.period === 'am' ? '🌅 Mattina' : '🌆 Pomeriggio'}</td>
                    <td class="max-w-xs truncate">${p.activity || '-'}</td>
                    <td>${p.manager_id}</td>
                    <td>
                        <button onclick="deletePlanningEntry(${p.id}, '${p.person}', '${p.date}', '${p.period}')" class="btn btn-danger">Elimina</button>
                    </td>
                </tr>
            `).join('');
            
            document.getElementById('planning-count').textContent = `Mostrate ${Math.min(items.length, 100)} di ${items.length} voci`;
        }

        function clearPlanningFilter() {
            document.getElementById('filter-date').value = '';
            loadPlanning();
        }

        async function loadNotes() {
            const res = await fetch(`${API_URL}/admin/notes`, { headers: adminHeaders() });
            const data = await res.json();
            
            const tbody = document.getElementById('notes-table');
            tbody.innerHTML = (data.notes || []).map(n => `
                <tr>
                    <td>${n.id}</td>
                    <td>${n.date}</td>
                    <td class="max-w-md truncate">${n.note}</td>
                    <td>${n.manager_name || n.manager_id}</td>
                    <td>
                        <button onclick="deleteNote(${n.id})" class="btn btn-danger">Elimina</button>
                    </td>
                </tr>
            `).join('');
        }

        // ============ MODAL & FORMS ============
        function showModal(title, formHtml) {
            document.getElementById('modal-title').textContent = title;
            document.getElementById('modal-form').innerHTML = formHtml;
            document.getElementById('modal').classList.remove('hidden');
        }

        function closeModal() {
            document.getElementById('modal').classList.add('hidden');
        }

        function showAddManager() {
            showModal('Nuovo Manager', `
                <input type="text" id="new-id" placeholder="ID (es. mario)" class="w-full" required>
                <input type="text" id="new-name" placeholder="Nome completo" class="w-full" required>
                <input type="password" id="new-password" placeholder="Password" class="w-full" required>
                <input type="color" id="new-color" value="#0891b2" class="w-full h-10">
                <label class="flex items-center gap-2">
                    <input type="checkbox" id="new-superuser"> Superuser (admin)
                </label>
                <button type="button" onclick="addManager()" class="btn btn-success w-full py-2">Crea Manager</button>
            `);
        }

        async function addManager() {
            const payload = {
                id: document.getElementById('new-id').value,
                name: document.getElementById('new-name').value,
                password: document.getElementById('new-password').value,
                color: document.getElementById('new-color').value,
                is_superuser: document.getElementById('new-superuser').checked
            };
            
            await fetch(`${API_URL}/admin/managers`, {
                method: 'POST',
                headers: { ...adminHeaders(), 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });
            
            closeModal();
            loadManagers();
        }

        function editManager(id) {
            const m = managersData.find(x => x.id === id);
            showModal('Modifica Manager', `
                <input type="text" value="${m.id}" disabled class="w-full bg-slate-100">
                <input type="text" id="edit-name" value="${m.name}" placeholder="Nome" class="w-full" required>
                <input type="text" id="edit-password" value="${m.password}" placeholder="Password" class="w-full" required>
                <input type="color" id="edit-color" value="${m.color}" class="w-full h-10">
                <label class="flex items-center gap-2">
                    <input type="checkbox" id="edit-superuser" ${m.is_superuser ? 'checked' : ''}> Superuser
                </label>
                <button type="button" onclick="updateManager('${m.id}')" class="btn btn-primary w-full py-2">Salva</button>
            `);
        }

        async function updateManager(id) {
            const payload = {
                name: document.getElementById('edit-name').value,
                password: document.getElementById('edit-password').value,
                color: document.getElementById('edit-color').value,
                is_superuser: document.getElementById('edit-superuser').checked
            };
            
            await fetch(`${API_URL}/admin/managers/${id}`, {
                method: 'PUT',
                headers: { ...adminHeaders(), 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });
            
            closeModal();
            loadManagers();
        }

        async function deleteManager(id) {
            if (!confirm(`Eliminare il manager ${id}? Verranno rimosse anche le persone del team.`)) return;
            
            await fetch(`${API_URL}/admin/managers/${id}`, {
                method: 'DELETE',
                headers: adminHeaders()
            });
            
            loadManagers();
            loadTeams();
        }

        function showAddPerson() {
            const options = managersData.map(m => `<option value="${m.id}">${m.name}</option>`).join('');
            showModal('Nuova Persona', `
                <input type="text" id="new-person-name" placeholder="Nome persona" class="w-full" required>
                <select id="new-person-manager" class="w-full">${options}</select>
                <button type="button" onclick="addPerson()" class="btn btn-success w-full py-2">Aggiungi</button>
            `);
        }

        async function addPerson() {
            const payload = {
                person_name: document.getElementById('new-person-name').value,
                manager_id: document.getElementById('new-person-manager').value
            };
            
            await fetch(`${API_URL}/admin/teams`, {
                method: 'POST',
                headers: { ...adminHeaders(), 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });
            
            closeModal();
            loadTeams();
        }

        function editPerson(name, managerId) {
            const options = managersData.map(m => 
                `<option value="${m.id}" ${m.id === managerId ? 'selected' : ''}>${m.name}</option>`
            ).join('');
            
            showModal('Modifica Persona', `
                <input type="hidden" id="edit-old-name" value="${name}">
                <input type="hidden" id="edit-old-manager" value="${managerId}">
                <input type="text" id="edit-person-name" value="${name}" placeholder="Nome" class="w-full" required>
                <select id="edit-person-manager" class="w-full">${options}</select>
                <button type="button" onclick="updatePerson()" class="btn btn-primary w-full py-2">Salva</button>
            `);
        }

        async function updatePerson() {
            const oldName = document.getElementById('edit-old-name').value;
            const oldManager = document.getElementById('edit-old-manager').value;
            const newName = document.getElementById('edit-person-name').value;
            const newManager = document.getElementById('edit-person-manager').value;
            
            // Prima rimuovi, poi aggiungi (per gestire cambio manager)
            await fetch(`${API_URL}/admin/teams/${encodeURIComponent(oldName)}?manager_id=${oldManager}`, {
                method: 'DELETE',
                headers: adminHeaders()
            });
            
            await fetch(`${API_URL}/admin/teams`, {
                method: 'POST',
                headers: { ...adminHeaders(), 'Content-Type': 'application/json' },
                body: JSON.stringify({ person_name: newName, manager_id: newManager })
            });
            
            // Se cambia nome, aggiorna anche il planning
            if (oldName !== newName) {
                await fetch(`${API_URL}/admin/teams/rename`, {
                    method: 'POST',
                    headers: { ...adminHeaders(), 'Content-Type': 'application/json' },
                    body: JSON.stringify({ old_name: oldName, new_name: newName })
                });
            }
            
            closeModal();
            loadTeams();
        }

        async function deletePerson(name, managerId) {
            if (!confirm(`Eliminare ${name} dal team?`)) return;
            
            await fetch(`${API_URL}/admin/teams/${encodeURIComponent(name)}?manager_id=${managerId}`, {
                method: 'DELETE',
                headers: adminHeaders()
            });
            
            loadTeams();
        }

        async function deletePlanningEntry(id, person, date, period) {
            if (!confirm('Eliminare questa voce del planning?')) return;
            
            await fetch(`${API_URL}/planning?person=${encodeURIComponent(person)}&date=${date}&period=${period}`, {
                method: 'DELETE'
            });
            
            loadPlanning();
        }

        async function deleteNote(id) {
            if (!confirm('Eliminare questa nota?')) return;
            
            await fetch(`${API_URL}/daily-notes/${id}`, {
                method: 'DELETE'
            });
            
            loadNotes();
        }

        function adminHeaders() {
            return {
                'X-Admin-Id': currentAdmin?.id || '',
                'X-Admin-Password': currentAdmin?.password || ''
            };
        }
    </script>
</body>
</html>
"""

@app.get("/admin", response_class=HTMLResponse)
async def admin_page():
    """Pagina admin HTML"""
    return ADMIN_HTML

# ============ ADMIN API ENDPOINTS ============

class AdminVerifyRequest(BaseModel):
    manager_id: str
    password: str

class ManagerCreateRequest(BaseModel):
    id: str
    name: str
    password: str
    color: str = "#0891b2"
    is_superuser: bool = False

class ManagerUpdateRequest(BaseModel):
    name: Optional[str] = None
    password: Optional[str] = None
    color: Optional[str] = None
    is_superuser: Optional[bool] = None

class PersonCreateRequest(BaseModel):
    person_name: str
    manager_id: str

class RenamePersonRequest(BaseModel):
    old_name: str
    new_name: str

def get_admin_credentials(request: Request):
    """Estrae credenziali admin dagli header"""
    admin_id = request.headers.get("X-Admin-Id", "")
    admin_password = request.headers.get("X-Admin-Password", "")
    return admin_id, admin_password

def require_superuser(request: Request):
    """Verifica che la richiesta sia da un superuser"""
    admin_id, admin_password = get_admin_credentials(request)
    if not verify_superuser(admin_id, admin_password):
        raise HTTPException(status_code=403, detail="Accesso negato. Solo superuser.")
    return True

@app.post("/api/admin/verify")
async def admin_verify(req: AdminVerifyRequest):
    """Verifica credenziali admin"""
    if not verify_superuser(req.manager_id, req.password):
        raise HTTPException(status_code=403, detail="Accesso negato")
    
    manager = db.get_manager_by_id(req.manager_id)
    return {"success": True, "name": manager['name']}

@app.get("/api/admin/managers")
async def admin_get_managers(request: Request):
    """Lista tutti i manager (admin only)"""
    require_superuser(request)
    managers = db.get_all_managers()
    return {"managers": managers}

@app.post("/api/admin/managers")
async def admin_create_manager(req: ManagerCreateRequest, request: Request):
    """Crea nuovo manager"""
    require_superuser(request)
    db.create_manager(
        id=req.id,
        name=req.name,
        password=req.password,
        color=req.color,
        is_superuser=1 if req.is_superuser else 0
    )
    return {"success": True}

@app.put("/api/admin/managers/{manager_id}")
async def admin_update_manager(manager_id: str, req: ManagerUpdateRequest, request: Request):
    """Aggiorna manager"""
    require_superuser(request)
    db.update_manager(
        id=manager_id,
        name=req.name,
        password=req.password,
        color=req.color,
        is_superuser=1 if req.is_superuser else 0 if req.is_superuser is not None else None
    )
    return {"success": True}

@app.delete("/api/admin/managers/{manager_id}")
async def admin_delete_manager(manager_id: str, request: Request):
    """Elimina manager"""
    require_superuser(request)
    db.delete_manager(manager_id)
    return {"success": True}

@app.get("/api/admin/teams")
async def admin_get_teams(request: Request):
    """Lista tutte le assegnazioni team"""
    require_superuser(request)
    assignments = db.get_all_team_assignments()
    return {"assignments": assignments}

@app.post("/api/admin/teams")
async def admin_add_person(req: PersonCreateRequest, request: Request):
    """Aggiunge persona a team"""
    require_superuser(request)
    db.add_person_to_team(req.person_name, req.manager_id)
    return {"success": True}

@app.delete("/api/admin/teams/{person_name}")
async def admin_remove_person(person_name: str, manager_id: str, request: Request):
    """Rimuove persona da team"""
    require_superuser(request)
    db.remove_person_from_team(person_name, manager_id)
    return {"success": True}

@app.post("/api/admin/teams/rename")
async def admin_rename_person(req: RenamePersonRequest, request: Request):
    """Rinomina persona nel planning"""
    require_superuser(request)
    # Aggiorna solo il planning (il team è già stato gestito)
    with db.get_connection() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "UPDATE planning SET person = ? WHERE person = ?",
            (req.new_name, req.old_name)
        )
    return {"success": True}

@app.get("/api/admin/planning")
async def admin_get_planning(request: Request, date: Optional[str] = None):
    """Lista planning (con filtro data opzionale)"""
    require_superuser(request)
    planning = db.get_all_planning()
    
    if date:
        planning = [p for p in planning if p['date'] == date]
    
    return {"planning": planning}

@app.get("/api/admin/notes")
async def admin_get_notes(request: Request):
    """Lista tutte le note"""
    require_superuser(request)
    notes = db.get_all_daily_notes()
    return {"notes": notes}


# ============ SERVE FRONTEND ============

# Se esiste index.html nella stessa cartella, servilo
FRONTEND_PATH = os.path.join(os.path.dirname(__file__), "index.html")

@app.get("/app", response_class=HTMLResponse)
async def serve_frontend():
    """Serve il frontend React"""
    if os.path.exists(FRONTEND_PATH):
        with open(FRONTEND_PATH, 'r', encoding='utf-8') as f:
            return f.read()
    return HTMLResponse("<h1>Frontend non trovato</h1><p>Posiziona index.html nella stessa cartella di main.py</p>")


if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)
